import numpy as np
import scipy as sp
import networkx as nx


def neumann(n):
    """Returns a Von Neumann neighborhood graph and position dictionary. 
    
    n : number of nodes
    """
    dim = int(np.sqrt(n))
    G = nx.grid_2d_graph(dim, dim)
    pos = {i: i for i in G.nodes}
    return G, pos


def moore(n):
    """Returns a Moore neighborhood graph and position dictionary.
    
    n : number of nodes
    """
    dim = int(np.sqrt(n))
    G = nx.grid_2d_graph(dim, dim)

    # create additional diagonal edges 
    # that are not already included in Neumann graph
    diagonals = []
    for x in range(dim - 1):
        for y in range(dim - 1):
            diagonals.append(((x, y), (x + 1, y + 1)))
            diagonals.append(((x + 1, y), (x, y + 1)))

    G.add_edges_from(diagonals)
    pos = {i: i for i in G.nodes}
    return G, pos


def voronoi(n):
    """Returns a Voronoi neighborhood graph and position dictionary.
    
    n : number of nodes
    """
    pos = {i: (np.random.random(), np.random.random()) for i in range(n)}
    tri = sp.spatial.Delaunay(list(pos.values()))

    # create an edgelist from a list of simplices
    # simplices are closed triangles create by Delaunay triangulation
    edges = []
    for s in tri.simplices:
        for i in range(len(s)):
            j = (i + 1) % len(s)
            e = (s[i], s[j])
            edges.append(e)

    G = nx.Graph()
    G.add_edges_from(edges)
    return G, pos


def social(n):
    """Returns a Watts Strogatz graph with average degree 10 and rewiring probability 0.01.
    This corresponds to realistic social network structure of scientific collaborator network 
    with high average clustering and low shortest average path length
    
    n : number of nodes
    """
    
    G = nx.watts_strogatz_graph(n, 10, 0.01)
    pos = nx.spring_layout(G)
    return G, pos