import numpy as np


class Agent:
    """Social agent equipped with mechanisms of epistemic vigilance.

    model: reference to model class
    id : unique node on network graph
    m : message integer 
    b : belief integer 
    i : interest integer
    h : boolean happy
    """
    def __init__(self, model, index, message, belief, interest):
        self.model = model
        self.id = index
        self.m = message # -1, 0, 1
        self.b = belief  # -1, 0, 1
        self.i = interest   #  0, 1
        self.h = False 


    def neighbors(self):
        """Returns all neighbor agents"""
        ids = list(self.model.G.neighbors(self.id))
        agents = [a for a in self.model.agents if a.id in ids]
        return agents


    def crival(self, sre):
        """Agent mechanism that enables agent to perform critical evaluation.

        sre : success rate of critical evaluation
        """
        # meet random interaction partner
        nbors = self.neighbors()
        other = np.random.choice(nbors)

        if other.m != 0:
            # if no background b then m gets accepted
            if self.b == 0:
                self.m = other.m
            
            # if matching background b then m gets accepted
            elif self.b == other.m:
                self.m = other.m
            
            # if different background b then m gets evaluated
            else:
                if np.random.random() < sre:
                    # successful critical evaluation means that
                    # self finds convincing evidence for the truth
                    self.m = 1
                    self.b = 1
                else:
                    # unsuccessful critical evaluation means that
                    # self finds convincing evidence for the lie
                    self.m = -1
                    self.b = -1


    def plaucheck(self, sre, srp):
        """Agent mechanism that enables agent to perform plausibility checking. 
        Includes critical evaluation.

        sre : success rate of critical evaluation
        srp : success rate of plausibility checking
        """
        # meet random interaction partner
        nbors = self.neighbors()
        other = np.random.choice(nbors)

        if other.m != 0: 
            # if no background b then m gets accepted
            if self.b == 0:
                self.m = other.m
            
            # if matching background b then m gets accepted
            elif self.b == other.m:
                self.m = other.m
            
            # if different background b then m gets evaluated
            else:
                # if not interested then plausibility checking
                if self.i == 0:
                    if np.random.random() < srp:
                        # successful plausibility check means that 
                        # self is able to remove inconsistency by rejection
                        self.m = self.m
                    else:
                        # unsuccessful plausibility check means that
                        # self is able to remove inconsistency by acception
                        self.m = other.m
                        self.b = other.m
                # if interested then critical evaluation
                else:
                    if np.random.random() < sre:
                        # successful critical evaluation means that
                        # self finds convincing evidence for the truth
                        self.m = 1
                        self.b = 1
                    else:
                        # unsuccessful critical evaluation means that
                        # self finds convincing evidence for the lie
                        self.m = -1
                        self.b = -1   
        else:
            if other.m != 0:
                raise ValueError('happening {}'.format(other.m))    


    def compcheck(self, sre, srp, src):
        """Agent mechanism that enables agent to perform competence checking.
        Includes critical evaluation.
        Includes plausibility checking.

        sre : success rate of critical evaluation
        srp : success rate of plausibility checking
        src : success rate fo competence checking
        """
        # meet random interaction partner
        nbors = self.neighbors()
        other = np.random.choice(nbors)

        if other.m != 0:
            # if no background b then m gets accepted
            if self.b == 0:
                self.m = other.m
            
            # if matching background b then m gets accepted
            elif self.b == other.m:
                self.m = other.m
            
            # if different background b then m gets evaluated
            else:

                # if not interested agents wants check plausibility
                if self.i == 0:

                    # but if other is interested then chance for competence checking
                    if other.i == 1:
                        if np.random.random() < src:
                            # successful competence check means that
                            # self gets convinced by other
                            self.m = other.m
                            self.b = other.m
                        else:
                            # unsuccessful competence check means that
                            # self falls back on plausibility checking
                            if np.random.random() < srp:
                                # successful plausibility check means that 
                                # self is able on remove inconsistency by rejection
                                self.m = self.m
                                
                            else:
                                # unsuccessful plausibility check means that
                                # self is able to remove inconsistency by acception
                                self.m = other.m
                                self.b = other.m

                    # if other is not interested then simply plausibility checking
                    else:
                        if np.random.random() < srp:
                            # successful plausibility check means that 
                            # self is able to remove inconsistency by rejection
                            self.m = self.m
                        else:
                            # unsuccessful plausibility check means that
                            # self is able to remove inconsistency by acception
                            self.m = other.m
                            self.b = other.m

                # if interested then critical evaluation
                else:
                    if np.random.random() < sre:
                        # successful critical evaluation means that
                        # self finds convincing evidence for the truth
                        self.m = 1
                        self.b = 1
                    else:
                        # unsuccessful critical evaluation means that
                        # self finds convincing evidence for the lie
                        self.m = -1
                        self.b = -1 