from mesa import Agent
import random
import networkx as nx
from pytz import NonExistentTimeError
from argument import WeightedArgument
from enum_states import State
import numpy as np


class DebateAgent(Agent):
    """
    An agent participating in a debate.
    """

    def __init__(self, unique_id, model, number_of_agents_on_first_issue, GPOV1, GPOV2,
                 bias=0):
        """
        Creates a new agent

        Args:
            GPOV1 and GPOV2 : the agent's own representation of the debate Together they correspond to the view of the agent.
            preferred GPOV : the agent preferred Research Program.
            name: name of the agent
            state: state of the agent, it can either be busy or free.
            bias: how much the agent is biased.


        """
        super().__init__(unique_id, model)
        self.number_of_agents_on_first_issue = number_of_agents_on_first_issue
        self.GPOV1 = GPOV1
        self.GPOV2 = GPOV2
        self.opinion = None
        self.preferred_GPOV = None
        self.current_arg = None  # the argument the agent is attacking when busy studying
        self.result = None  # the argument the agent has produced 
        self.name = unique_id
        self.state = State.FREE
        self.bias = bias


    def get_opinion(self, semantic):
        self.opinion = semantic.get_argument_value(self.opinion_graph.get_issue(), self.opinion_graph)
        return self.opinion

    def __str__(self) -> str:
        return str(self.name)

    def __repr__(self):
        return self.__str__()

    def update(self, new_arguments):
        '''
        the method allows an agent to add a list of new arguments to her view, i.e. to her two representations
        of the research programs GPOV1 and GPOV2
        '''

        # adding the new arguments the agent's woldview
        for arg in new_arguments:
            tree = arg.tree
            tree_name, tree_gpov_name = tree.get_name(), tree.get_gpov_name()
            # print(tree_name, tree_gpov_name)
            if tree_gpov_name == '1':
                self.GPOV1.add_argument(arg.id, tree_name, arg.attacked_arg.id)
            if tree_gpov_name == '2':
                self.GPOV2.add_argument(arg.id, tree_name, arg.attacked_arg.id)

        return None

    def learn(self, arg):
        '''the method allows the agent to add an argument to their view'''
        if arg.tree.get_gpov_name() == '1':
            self.model.GPOV1.publish_argument(arg)
        else:
            self.model.GPOV2.publish_argument(arg)

    def get_scores(self):
        score1 = self.GPOV1.get_score()
        score2 = self.GPOV2.get_score()
        return score1, score2

    def choose(self):
        '''
        The agent chooses a Research Program, and then chooses a node. Then, she starts investigating that node. She is now busy.
        '''

        # choosing a GPOV, then a random node and becoming busyS
        if self.state == State.FREE:
            
            # choosing a GPOV
            if self.preferred_GPOV is None: # initialization round
                if self.name < self.number_of_agents_on_first_issue:
                    self.preferred_GPOV = 1
                elif self.name >= self.number_of_agents_on_first_issue:
                    self.preferred_GPOV = 2

            else:
                score1 = self.GPOV1.get_score()
                score2 = self.GPOV2.get_score()
                if score1 > score2:
                    self.preferred_GPOV = 1
                    # print("GPOV1 has score ", score1, "GPOV2 score", score2, ", chose 1")
                elif score2 > score1:
                    self.preferred_GPOV = 2
                    # print("GPOV1 has score ", score1, "GPOV2 score", score2, ", chose 2")
        
            arg = random.choice(self.all_arguments())
            self.current_arg = self.model.find_arg_by_id(arg.id)

            self.state = State.BUSY_S
            
        return None

    def all_arguments(self):
        '''Returns all the arguments the agent is aware of.'''
        return self.GPOV1.all_arguments() + self.GPOV2.all_arguments()

    def is_opposed(self, arg):
        # checks whether the argument goes against the agent's preferred GPOV
        o = arg.tree.get_oddity(arg)
        if str(arg.tree.get_gpov_name()) == "1":
            if str(self.preferred_GPOV) == "1":
                if o == 0:
                    return True
                elif o == 1:
                    return False
            elif str(self.preferred_GPOV) == "2":
                if o == 0:
                    return False
                elif o == 1:
                    return True
        elif str(arg.tree.get_gpov_name()) == "2":
            if str(self.preferred_GPOV) == "1":
                if o == 0:
                    return False
                elif o == 1:
                    return True
            elif str(self.preferred_GPOV) == "2":
                if o == 0:
                    return True
                elif o == 1:
                    return False

    def study(self):
        '''
        This method allows the agent to study her current argument. If she does not find an attack,
        she stops studying and becomes free.'''

        if self.state == State.BUSY_S: 

            strength = self.try_generate_attack(self.current_arg)
            if strength is not None:
                self.result = WeightedArgument(strength, self.current_arg, self.current_arg.tree)
                self.state = State.SUCCESS_S
                
            else :
                # the agent abandons the target and becomes free
                self.clear_state()
                self.state = State.FREE
            
        return None

    def publish(self):

        if self.state == State.SUCCESS_S:
            # the agent publishes
            self.learn(self.result)
            self.model.new_arguments += [(self, self.result)]
            self.clear_state()
            self.state = State.FREE

        return None

    def clear_state(self):
        # clearing all variables
        self.current_arg = self.result = None

    def try_generate_attack(self, arg):

        '''The method allows an agent to attempt to attack the argument that is passed as an input.
        First the method checks if the argument is opposed to the agent's preference.
        Then, it computes the probability for the agent to find an attack, and toss the respective coin.
        Finally, if an argument has been found, it computes the strength of such an argument and returns it.
        '''

        # checking if the argument is opposite to the agent's preference
        tree = arg.tree
        tree_gpov_name = tree.get_gpov_name()
        o = tree.get_oddity(arg)
        if str(tree_gpov_name) == "1":
            if str(self.preferred_GPOV) == "1":
                if o == 0:
                    position = "opp"
                elif o == 1:
                    position = "not opp"
            elif str(self.preferred_GPOV) == "2":
                if o == 0:
                    position = "not opp"
                elif o == 1:
                    position = "opp"

        elif str(tree_gpov_name) == "2":
            if str(self.preferred_GPOV) == "1":
                if o == 0:
                    position = "not opp"
                elif o == 1:
                    position = "opp"
            elif str(self.preferred_GPOV) == "2":
                if o == 0:
                    position = "opp"
                elif o == 1:
                    position = "not opp"

        if position == "opp":

            proba = min(1, 1 - arg.strength  + self.bias)
            # print("biased against the argument. Proba :", proba)
        else:
            proba = max(0, 1 - arg.strength - self.bias)
            # print("biased in favor of the argument. Proba :", proba)
        if random.uniform(0, 1) <= proba:
            # generate strength with distribution of mean argument strength
            mu, sigma = 1 - arg.strength, self.model.std_attacks  # mean and standard deviation
            s = np.random.normal(mu, sigma)
            if s <= 0:
                return 0
            return min(s, 1)
    
