from agents import DebateAgent
from graphs import DebateGraph
from mesa import Model
from schedule import SimultaneousDebateActivation
from datetime import datetime
import random
from gpov import GPOV, WeightedGPOV
from mesa.datacollection import DataCollector


class Debate(Model):
    """
    Model implementing the protocol 
    """

    description = (
        "A model for simulating debates"
    )

    def __init__(self, param_dict):
        
        """
        Creates a new model with the given parameters.

        Args: 
            num_agents = total number of agents
            num_issues = number of issues by ResProg
            initial_support = initial support for ResProg 1
            new_arguments = the arguments that are published in the present turn.
            p_see = probability for an agent to see a new argument and include it into its
            research program.
            issue_mean_1 = The strength of research program 1
            issue_mean_2 = The strength of research program 2
            std_attacks = the standard deviation of the distribution for sampling the strength of the attacks.
            bias = the bias each agent has.
            shared_beliefs = the intensity of the shared beliefs

                """

                
        super().__init__()
        # Set parameters
        self.initial_support = param_dict['initial_support']
        self.num_agents = param_dict['num_agents']
        self.num_issues = param_dict['num_issues']
        self.new_arguments = []
        self.p_accept = param_dict['p_accept']
        self.schedule = SimultaneousDebateActivation(self)
        self.time = str(datetime.now()).replace('.', '_').replace(':',
                                                                  '_')  # a time marker to help save all relevant informations
        self.arg_count = 0

        self.issue_mean_1 = param_dict['issue_mean_1']
        self.issue_mean_2 = param_dict['issue_mean_2']
        self.std_attacks = param_dict['std_attacks']
        self.bias = param_dict['bias']
        self.shared_beliefs = param_dict['shared_beliefs']       

        print("=============== MODEL INITIALIZATION =========================================")
        print()

        '''
        First, we initialize the Research Programs - called GPOV here. These are "general" research programs that collect all the arguments that are 
        published at any point. The View of each agent is a subset of these two research programs. '''


        # creating the research programs 

        if self.issue_mean_1 is not None:
            issue_strength_1 = [self.issue_mean_1 for i in range(self.num_issues)]
            issue_strength_2 = [self.issue_mean_2 for i in range(self.num_issues)]
        else:
            issue_strength_1 = [random.randint(0, 10) for i in range(self.num_issues)]
            issue_strength_2 = [random.randint(0, 10) for i in range(self.num_issues)]
        self.GPOV1 = WeightedGPOV('1', issue_strength_1)
        self.GPOV2 = WeightedGPOV('2', issue_strength_2)

        # creating the agents

        count_agent = 0
        for i in range(self.num_agents):
            agent_GPOV1 = GPOV('1', self.GPOV1.get_issue_ids())
            agent_GPOV2 = GPOV('2', self.GPOV2.get_issue_ids())
            agent = DebateAgent(count_agent, self, self.initial_support, agent_GPOV1, agent_GPOV2, bias=self.bias,
                                # max_reviewing=self.max_reviewing
                                )
            self.schedule.add(agent)
            count_agent += 1
        
        assert count_agent == self.num_agents

    def all_arguments(self):
        ''' returns all the arguments that have been published so far. '''
        return self.GPOV1.all_arguments() + self.GPOV2.all_arguments()

    def find_arg_by_id(self, id):
        '''returns the argument using its id'''
        res = self.GPOV1.find_id(id)
        if res is None:
            res = self.GPOV2.find_id(id)
        return res

    def step(self, i):
        '''advances the model of one step'''
        self.current_step = i
        self.schedule.step()



    def run_model(self, step_count=40):
        '''
        this method runs the model for a number of steps which is given as an input. It also collects the data at each step.
        Intuitively, on top the parameters input in the simulation, the stats also collect the amount of people preferring ResProg 1 and 2
        at each step.
        We also collect stats about the arguments produced during the simulation
        '''

        stats = []

        for i in range(step_count):
            self.step(i)
            step_stats = dict()
            step_stats['STEP'] = i
            step_stats['Strength 1'] = self.issue_mean_1
            step_stats['Strength 2'] = self.issue_mean_2
            step_stats['Bias'] = self.bias
            step_stats['Shared Beliefs'] = self.shared_beliefs
            # step_stats['Bias A'] = self.bias_a
            step_stats['STD of Attacks'] = self.std_attacks
            step_stats['Initial Support for ResProg 1'] = self.initial_support
            prefer_1 = 0
            prefer_2 = 0
            for agent in self.schedule.agents:
                if str(agent.preferred_GPOV) == '1':
                    prefer_1 += 1
                else:
                    prefer_2 += 1
                
            step_stats['Prefer 1'] = prefer_1
            step_stats['Prefer 2'] = prefer_2

            step_stats['SOTA Score 1'] = self.GPOV1.get_score()
            step_stats['SOTA Score 2'] = self.GPOV2.get_score()
            step_stats['Num Issues'] = self.num_issues
            step_stats['P Accept'] = self.p_accept
            step_stats['Num Agents'] = self.num_agents

            stats += [step_stats]

        
        print("====================================== Debate Over ============================================")

        # getting the argument stats

        self.all_arg_stats = []

        for tree in self.GPOV1.graphs + self.GPOV2.graphs:
            acceptable_arguments = tree.get_acceptable_arguments()
            for arg in tree.nodes:
                arg_stats = dict()
                arg_stats['id'] = arg.id
                arg_stats['GPOV'] = tree.GPOV.name
                arg_stats['Tree'] = tree.name
                arg_stats['strength'] = arg.strength
                # pro or con
                arg_stats['Pro/Con'] = 'PRO' if tree.get_oddity(arg)==1 else 'CON'
                # status : accepted or not ?
                arg_stats['Label'] = 1 if (arg in acceptable_arguments) else 0
                self.all_arg_stats += [arg_stats]

        return stats

