extensions [matrix]

; DECLARE GLOBAL VARIABLES

globals [
  ; further ratios can be calculated from these.
  problems-solved-results
  problems-failed-results
  problems-unattempted-results

  problem-elements ; set of values a problem could be

  problem-matrix
  prob-info-matrix
  turtle-adj-matrix

  action-energy-infinite

  scoring-clean-start
  total-score
  tick-solves
  tick-score

]

breed [workers worker]
breed [managers manager]

; * Set up agents
turtles-own [
  my-strategy
  my-weight-others-input
  my-problem-list
  my-last-problem
  my-problem-counter
  my-solution
  my-solves
  my-fails
  my-unattempts
  my-mem-length
  my-mem
  my-input-from-others
  info-to-transfer
  comm-connections-left
  ; set my-probs list (item [who] problem-list) ; will be used later in the go func
]

; * set up world.
; - Size of world.
; - Number of agents.



to setup
  clear-all

  resize-world 0 (problems-per-worker - 1) 0 (number-of-workers - 1)
  ; tick on which the starting garbage is flushed out safely
  set scoring-clean-start (10 + agent-mem-length)
  if verbose? [ print (word "scoring-clean-start: " scoring-clean-start) ]

  setup-problems

  if (visualization?) [viz-problems]

  create-workers number-of-workers [
    ;set size (world-width / number-of-workers)
    set color black
    set my-problem-counter 0
    set my-mem-length agent-mem-length

    ; mode:
    set my-input-from-others (list one-of problem-elements)

    set breed workers

    set my-mem n-values my-mem-length [1] ; FOR TESTING
    ;set my-mem reduce sentence n-values my-mem-length [i -> n-of 1 problem-elements]
    ; although really it's garbage whether it's random or all 1s, doesn't matter

    ; for mode setting:
    set info-to-transfer [1]
    set comm-connections-left comm-connections-max

  ]

  create-managers number-of-managers [
    set color black ; to make trails a little easier to see
    set shape "star"
    set my-problem-counter 0
    set my-mem-length agent-mem-length
    set breed managers
    set ycor (ycor + (max-pycor / 3 ))

    ; mode:
    set my-input-from-others (list one-of problem-elements)

    ; set memory to a memory-length list of random elements from the list of possible ones
    set my-mem n-values my-mem-length [3] ; FOR TESTING

    ;set my-mem reduce sentence n-values my-mem-length [i -> n-of 1 problem-elements]
    ; although really it's garbage whether it's random or all 1s, doesn't matter

    ; for mode setting:
    set info-to-transfer [3]
    set comm-connections-left comm-connections-max
  ]

  put-workers-on-lanes
  set-first-strats
  make-adj-matrix

  ; initialize result lists
  set problems-solved-results [] ; problematic category if partial success allowed
  set problems-failed-results [] ; problematic category if partial success allowed
  set problems-unattempted-results []

  set action-energy-infinite TRUE

  if verbose? [ print word "turtle-adj-matrix: " turtle-adj-matrix ]


  reset-ticks
end

to setup-problems

  ; IMPORTANT: problem-elements must be ints ONLY

  ifelse (problem-type = "binary") [
    set problem-elements [ 1 2 ]
  ]
  [
    set problem-elements [ 1 2 ]
  ]

  ; TODO: Size of matrix should be LONGER than problems per worker to allow for the flushing
  ; of the random junk that the workers start with.
  ; I believe it's 3 + memory ticks, but 10 + memory ticks "to be safe."

  set problem-matrix (matrix:make-constant problems-per-worker number-of-workers
    item 0 problem-elements)

  ; RANDOMIZED PROBLEM MATRIX FOR TESTING
  ;set problem-matrix (fill-matrix problems-per-worker number-of-workers [
  ;  -> one-of problem-elements])

  ; ALL TWOS MATRIX FOR TESTING
  ;set problem-matrix (matrix:make-constant problems-per-worker number-of-workers 2)

  ; ALL ONE RANDOM ELEMENT MATRIX FOR TESTING
  ;set problem-matrix (matrix:make-constant problems-per-worker number-of-workers
  ;  item 2 problem-elements)

  ; Find how many usable rows there are.
  ; Take problems-per-worker, use the tick on which the starting garbage is flushed out
  ; safely to find the number of usable rows for the intervention.

  let number-usable-rows round (problems-per-worker - scoring-clean-start)

  if verbose? [print( word "number-usable-rows: " number-usable-rows) ]

  ; Environment. Always use 50% one value, 50% another (if time, I can use more than 2 values):

  if env-pattern = "custom"
  [ ; Make whole matrix same value
    ; based on 3 transitions, 1 stagger
    set problem-matrix (matrix:make-constant problems-per-worker number-of-workers
      (item 0 problem-elements))

    if verbose? [ print word "number-usable-rows: " number-usable-rows ]
    ;note: number-usable-rows round (problems-per-worker - scoring-clean-start)
    let nth-of-usable (floor (number-usable-rows / (transitions + 1)))

    ; if it's not a clean division, get the remainder and append it to the last nth later
    let the-remainder remainder number-usable-rows (transitions + 1)
    if verbose? [ print word "the remainder: " the-remainder ]

    if verbose? [ print(word "nth-of-usable: " nth-of-usable) ]
    let switch-row (scoring-clean-start)
    ;let switch-row (scoring-clean-start + nth-of-usable)
    if verbose? [ print(word "switch-row: " switch-row) ]

    let trans-counter 0
    let colorit? TRUE
    let list-of-rows-to-change []
    while [trans-counter < transitions]
    [
      if colorit? [
      let list-rows n-values (nth-of-usable) ; n-values size reporter
        ; Reports a list of length size containing values computed by repeatedly running the reporter.
        [ i -> i + (switch-row + (nth-of-usable * (trans-counter + 1))) ] ; move nth up to clean and lead start

        if verbose? [ print (word "list-rows: " list-rows) ]
        set list-of-rows-to-change (sentence list-of-rows-to-change list-rows)

        ; deal with the remainder, append it to the last nth
        if (the-remainder > 0 and (trans-counter = (transitions - 1))) [
          if verbose? [ print "Remainder appending if-function is being called." ]
          let remainder-rows n-values (the-remainder)
          [i -> i + (switch-row + (nth-of-usable * (trans-counter + 2)))]
          if verbose? [ print word "Remainder rows: " remainder-rows ]
          set list-of-rows-to-change (sentence list-of-rows-to-change remainder-rows)
        ]
      ]
      ifelse (colorit?) [set colorit? FALSE] [set colorit? TRUE]
      set trans-counter trans-counter + 1
    ]

    if verbose? [ print (word "list-of-rows-to-change: " list-of-rows-to-change ) ]
    foreach list-of-rows-to-change [i -> matrix:set-row problem-matrix i (
      n-values number-of-workers [(item 1 problem-elements)] )
    ]

    if (stagger > 0 ) [desyncifier stagger]
    ; if (noisy-env > 0) [env-noisifier nois-env]

  ]

  if ( oppositizer > 0 ) [oppositize]
  ; removing randomness from row chosen so I can group landscapes in the analysis phase
  ; I don't believe having random rows vs not oppositized will affect results

  if verbose? [ print problem-matrix ]
  if verbose? [ print matrix:pretty-print-text problem-matrix ]

  set prob-info-matrix matrix:copy problem-matrix ; new copy containing data

  ; can add "noise" by randomly choosing some % of cells and changing them

end

to oppositize
  let number-cols (round (oppositizer * number-of-workers))
  if verbose? [ print(word "number-cols: " number-cols ) ]

  ; make list of workers, choosing each item/worker at random an oppositizer number of times
  let agent-list n-of number-cols (n-values number-of-workers [ i -> i])

  if verbose? [ print(word "agent-list: " agent-list ) ]

  ; foreach item in that list
  foreach agent-list
    [ z -> let col-index z
      let prob-col-list matrix:get-column problem-matrix col-index
      if verbose? [ print(word length prob-col-list " prob-col-list: " prob-col-list) ]
      let new-col []

      foreach prob-col-list
        [ x -> ifelse x = 1
          [
            if verbose? [ print(word "x: " x) ]
            set new-col lput 2 new-col
            if verbose? [ print(word length new-col " new-col: " new-col) ]
          ]
          [
            if verbose? [ print(word "x: " x) ]
            set new-col lput 1 new-col
            if verbose? [ print(word length new-col " new-col: " new-col) ]
          ]
        ]

      if verbose? [ print(word length new-col " new-col: " new-col) ]

      ; Write back to matrix
      matrix:set-column problem-matrix col-index new-col
    ]
end

to desyncifier [ stagger-num ]
  ; Desync/stagger matrix.
  ; Make list that's index of every problem-matrix col, same as the number-of-workers.
  let col-index-list n-values number-of-workers [ i -> i ]
  if verbose? [ print (word "col-index-list: " col-index-list) ]

  let stagger-list map [ i -> i + stagger-num ] col-index-list
  if verbose? [ print (word "stagger-list: " stagger-list) ]

  foreach col-index-list [
    z -> let col-index z

    let prob-col-list matrix:get-column problem-matrix col-index
    if verbose? [ print(word length prob-col-list " prob-col-list: " prob-col-list) ]

    ;make sublist without trash
    let trashless-list sublist prob-col-list scoring-clean-start (length prob-col-list)
    if verbose? [ print (word length trashless-list " trashless-list: " trashless-list ) ]

    ; make sublist
    let front-probs sublist trashless-list 0 (item col-index stagger-list)
    if verbose? [ print (word length front-probs " front-probs: " front-probs ) ]

    ;sentence the last bits back on
    let list-to-trim sentence trashless-list front-probs
    if verbose? [ print (word length list-to-trim " list-to-trim: " list-to-trim) ]

    ;trim the item x number of items off the front of the prob-col-list
    let new-col-list sublist list-to-trim ; provide first and second position below
      (item col-index stagger-list) (length list-to-trim) ;had +1 here originally
    if verbose? [ print (word length new-col-list " new-col-list: " new-col-list) ]

    ; put the trash back on the front.
    let trash-list n-values scoring-clean-start [item 0 problem-elements]
    if verbose? [ print(word length trash-list " trash-list: " trash-list) ]

    let new-col-plus-trash sentence trash-list new-col-list

    ; Write back to matrix
    matrix:set-column problem-matrix col-index new-col-plus-trash

  ]

  ; Verify that every column has the same number of values
  if verbose? [
        foreach col-index-list [y -> print(word "Sum of col " y " is "
          reduce + matrix:get-column problem-matrix y )]
  ]

end

to-report fill-matrix [n m generator]
  report matrix:from-row-list n-values n [n-values m [runresult generator]]
  ; the generator "variable" is actually a function passed to this function
end

to-report landscape
  ; for BehaviorSpace so I can output the problem landscape to CSV and read in to process
  report matrix:to-row-list problem-matrix
end

to put-workers-on-lanes
  let y-counter 0

  foreach sort turtles [ i -> ask i [
    setxy min-pxcor (min-pycor + y-counter)
    set heading 90
    ]
   set y-counter y-counter + 1
  ]

  if count managers > 0 [
    ask managers [
      set ycor mean [pycor] of turtles ; if manager, ask manager move-to somewhere
      forward -1
    ]
  ]

end

to set-first-strats

  ask turtles [
    set my-weight-others-input weight-others-input

    if decision-strategy = "mode with input"
      [ set my-strategy "mode with input" ]
  ]

end

to make-adj-matrix
  set turtle-adj-matrix matrix:make-identity (count turtles)
  let mgr-list ([who] of managers)

  ; Fills 1s for workers-managers communication
  foreach mgr-list [
    x -> matrix:set-row turtle-adj-matrix x (n-values (count turtles) [1])]
  foreach mgr-list [
    x -> matrix:set-column turtle-adj-matrix x (n-values (count turtles) [1])]

end

to viz-problems
  ask patches [
    let my-val (matrix:get problem-matrix pxcor pycor)
    ifelse (my-val = 1) [set pcolor blue + 2 ][set pcolor white]
  ]
end

to communicate-among-agents

  ; In the last decision-making round, workers stored to memory what the actual problem was.
  ; Managers stored to memory what the consolidation of all their inputs was.
  ; Workers and managers "recall" the last rounds decision outcome.
  ; They can use a recall-type of "most recent" and "mode".

  ask turtles [
    if (recall-type = "most recent") [
      set info-to-transfer (list first my-mem)
      if verbose? [ print (word "Turtle " who " here, most recent mem is: " info-to-transfer) ]
    ]

    ; not currently enabled or used
    if (recall-type = "mode") [
      set info-to-transfer (list weigh-and-take-mode my-mem my-input-from-others 0)
      if verbose? [ print (word "Turtle " who " here, took the mode of my-mem: " info-to-transfer) ]
    ]

    set my-input-from-others []
  ]

  ask workers [
    let comm-list-all matrix:get-row turtle-adj-matrix who ;get self's row from adjmatrix
    let comm-list find-nonzero-connects comm-list-all ;make list of turtles self can comm with
    let comm-list-agents filter [ i -> i != who ] comm-list ; remove self from list
    if verbose? [ print (word "Worker " who " here, I can comm with these turtles: " comm-list-agents ".") ]

    let new-input info-to-transfer

    ; if comm-connects-unlimited = TRUE, comm-connections-max should be 1.
    foreach comm-list-agents [
      x -> ask turtle x [
        if (comm-connections-left > 0) [
          if verbose? [ print (word "Turtle " who " here, comm-connections-left > 0.") ]
          set my-input-from-others fput new-input my-input-from-others

          if (comm-connects-unlimited = FALSE ) [
            if verbose? [ print (word "Turtle " who " here, comm-connections-left is " comm-connections-left ".") ]
            set comm-connections-left (comm-connections-left - 1)
          ]
          if verbose? [ print (word "Turtle " who " here, now comm-connections-left is " comm-connections-left ".") ]
        ]
      ]
    ]
  ]

  ask managers [
    let comm-list-all matrix:get-row turtle-adj-matrix who
    let comm-list find-nonzero-connects comm-list-all
    let comm-list-agents filter [ i -> i != who ] comm-list ; remove self from list
    if verbose? [ print (word "Manager " who " here, I can comm with these turtles: " comm-list-agents ".") ]

    if verbose? [ print (word "Manager " who " here, info-to-transfer is " info-to-transfer ".") ]

    let new-input info-to-transfer ; this is just the mode, which the manager put into memory last round

    foreach comm-list-agents [
      x -> ask turtle x [
        if (comm-connections-left > 0) [
          if verbose? [ print (word "Turtle " who " here, comm-connections-left > 0.") ]
          set my-input-from-others fput new-input my-input-from-others

          if (comm-connects-unlimited = FALSE ) [
            if verbose? [ print (word "Turtle " who " here, comm-connections-left is " comm-connections-left ".") ]
            set comm-connections-left (comm-connections-left - 1)
          ]
          if verbose? [ print (word "Turtle " who " here, now comm-connections-left is " comm-connections-left ".") ]
        ]
      ]
    ]
  ]

  ask turtles [
    set comm-connections-left comm-connections-max
    if verbose? [ print (word "Turtle " who " here, comm-connections-left reset to " comm-connections-left ".") ]
  ]

end

to-report find-nonzero-connects [numbers-list]
  let ties filter [i -> i > 0 ] numbers-list
  report filter [i -> member? (item i numbers-list) ties]
    n-values (length numbers-list) [i -> i]

 ; From Seth Tisue, in an old post on the netlogo mailing list:
 ;what's happening is, first `n-values` gives a list of all the valid
 ;indices into the list, then we filter that list, keeping only the
 ;indices that point to items that are equal to the biggest number
 ;in the list.  And that's exactly the answer we're trying to compute.

end


to tackle-problem
  ask workers [
    ; retrieve any cues (some type of affordance that has info) about that problem.

    ; The problems have an info matrix that is the same size.
    if verbose? [ print (word "Worker " who " here, this problem looks like " matrix:get prob-info-matrix my-problem-counter who ".") ]
    ;right 45

    decide-solution
    store-problem-in-memory ; stores the actual problem. could expand to use perception of prob

  ]

  ask managers [
    decide-solution ; uses the same strategy as the worker to consolidate the inputs into one
    store-consolidation-in-memory

  ]


end

to decide-solution
  ; Test strategies
  if my-strategy = "random per tick"
    [ set my-solution (one-of problem-elements) ]

  if my-strategy = "random at start" [ ; TODO: check this is working properly
    if ticks = 0
    [ set my-solution (one-of problem-elements) ]
  ]

  if my-strategy = "last problem" [
    ifelse my-problem-counter = 0 [ ; if 0, no history, so set randomly
      set my-solution (one-of problem-elements)
    ]
    [
      set my-solution (matrix:get problem-matrix (my-problem-counter - 1) who)
    ]
  ]

  ; ONLY strategy actually used
  if my-strategy = "mode with input" [
    ifelse (ticks = 0) [
      set my-solution (one-of problem-elements)
    ]
    [
      ; function that weighs mem and input and calculates decision
      set my-solution weigh-and-take-mode consult-memory consult-others my-weight-others-input

      if verbose? [ print (word "Turtle " who " here, choosing solution " my-solution ".") ]
    ]
  ]

end

to store-problem-in-memory ; for workers

  ; detect current problem, put that at beginning of my-mem list.
  let current-problem (matrix:get problem-matrix my-problem-counter who)
  if verbose? [ print (word "Turtle " who " here, current-problem is now " current-problem ".") ]

  ; newest memory is always the left-side / beginning of the list
  set my-mem fput current-problem my-mem

  if (length my-mem) > my-mem-length [
    if verbose? [ print (word "Turtle " who " here, pre-trim my-mem is now " my-mem ".") ]
    if verbose? [ print (word "Turtle " who " here, my-mem-length is " my-mem-length ".") ]
    ; make sublist, first position (inclusive), second position (exclusive)
    set my-mem sublist my-mem 0 (my-mem-length)
  ]

  if verbose? [ print (word "Turtle " who " here, my-mem is now " my-mem ".") ]

end

to store-consolidation-in-memory ; for managers

  ; For testing only
  ;set my-mem fput my-solution my-mem

  ; I pass the mode function an empty list for memory and weigh others-input 100% to get mode
  ; Because I use it like this, managers spit out extra verbosity from the weigh-and-take-mode function
  let current-mode weigh-and-take-mode [] consult-others 1
  if verbose? [ print (word "Turtle " who " here, current-mode is " current-mode ".") ]

  set my-mem fput current-mode my-mem

  if (length my-mem) > my-mem-length [
    if verbose? [ print (word "Turtle " who " here, pre-trim my-mem is now " my-mem ".") ]
    if verbose? [ print (word "Turtle " who " here, my-mem-length is " my-mem-length ".") ]

    ; make sublist, first position (inclusive), second position (exclusive)
    set my-mem sublist my-mem 0 (my-mem-length)
    if verbose? [ print (word "Turtle " who " here, my-mem is " my-mem ".") ]
  ]

end

to-report consult-memory
  ; Each agent stores memory separately.
  ; The prime benefit is better transparency on what's going on.
  ; But it should also be useful for implementing memory disruptions within this function.

  ; What does consult-memory need to output?
  ; A list.

  report my-mem
end

to-report consult-others
  report my-input-from-others
  ; Noise could be introduced here.
end

to-report weigh-and-take-mode [mem others-input weighing]

  ; mem must be a list
  ; others-input must be a list
  ; weighing must be a number between 0 and 1.

  ; Create a frequency table as long as the problem-elements list.
  ; freqency is a function defined in this model below
  let mem-freq map [ i -> frequency i mem] problem-elements

  if verbose? [ print (word "Turtle " who " here, others-input is " others-input) ]

  let others-input-flat others-input
  ifelse empty? others-input
    [ set others-input-flat others-input ]
    [ set others-input-flat unnest others-input ]
  let others-freq map [ i -> frequency i others-input-flat] problem-elements
  if verbose? [ print (word "Turtle " who " here, others-input-flat is " others-input-flat ".") ]

  ; I'm scaling these as lists, NOT converting to matrices.
  let weight weighing  ; weight will be a number between 0 and 1

  ; e.g. weights of .1 and .9 would give .1*[3,3,1]+.9*[1,4,0]=[1.2, 3.9, .1]
  let weighed-mem-freq map [ i -> i * (1 - weight) ] mem-freq
  let weighed-others-freq map [ i -> i * weight ] others-freq
  if verbose? [ print (word "Turtle " who " here, my weighed-mem-freq is " weighed-mem-freq ".") ]
  if verbose? [ print (word "Turtle " who " here, my weighed-others-freq is " weighed-others-freq ".") ]

  ; add the two lists
  ; get the position of the largest number in the list
  let summed-freq (map + weighed-mem-freq weighed-others-freq) ;adds the lists together
  if verbose? [ print (word "Turtle " who " here, my summed-freq table is " summed-freq ".") ]

  let choice-index max-positions summed-freq ; finds index of most freq

  if verbose? [ print (word "Turtle " who " here, my choice-index has these positions: " choice-index ".") ]

  let the-mode item (one-of choice-index) problem-elements
  ; takes random entry from the list of modes, usually list has only 1 value.
  ; Thus if there are two modes, one is chosen at random.
  ; That shouldn't happen often, but perhaps I should count how often it does.
  ; It won't work without the one-of b/c item needs a number, not a list like [0]

  report the-mode

end

to-report unnest [ xs ]
  let ys reduce sentence xs
  report ifelse-value (reduce or map is-list? ys) [ unnest ys ] [ ys ]
end

to-report frequency [an-item a-list]
	report length (filter [ i -> i = an-item] a-list)
end

to-report max-positions [numbers]
  let biggest max numbers
  report filter [i -> (item i numbers) = biggest]
    n-values (length numbers) [i -> i]
  ; filter reporter list
  ; item index list

  ; From Seth Tisue, in an old post on the netlogo mailing list:
  ; what's happening is, first `n-values` gives a list of all the valid
  ; indices into the list, then we filter that list, keeping only the
  ; indices that point to items that are equal to the biggest number
  ; in the list.  And that's exactly the answer we're trying to compute.

end

to score-solutions

  ask workers [

    if ticks > scoring-clean-start [

      ifelse my-solution = (matrix:get problem-matrix my-problem-counter who) [
        set tick-solves (tick-solves + 1)
        set my-solves (my-solves + 1)

        if verbose? [ print (word "Turtle " who " here, I got it right.") ]
        ]

        [
        if verbose? [ print (word "Turtle " who " here, I got it wrong.") ]
        ]

      if verbose? [ print (word "Turtle " who " here, this problem actually is "matrix:get problem-matrix my-problem-counter who ".") ]

      set total-score 100 * ((sum [ my-solves ] of workers) /
                             (sum [ my-problem-counter - scoring-clean-start ] of workers))

      if verbose? [ print (word "fraction: " ((sum [ my-solves ] of workers) / (sum [ my-problem-counter ] of workers))) ]
      if verbose? [ print (word "total-score: " total-score) ]
      set tick-score 100 * (tick-solves / number-of-workers)

      if verbose? [ print (word "tick-solves: " tick-solves ) ]

    ]
  ]

  set tick-solves 0

  ; Total score at the end of the run is given in the "go" function

end

to adjust-weights-if-needed [ real-problem-val ]
  if ticks > scoring-clean-start [
    ifelse my-solution != real-problem-val [
      if verbose? [ print (word "Turtle " who " here, I got it wrong, I will adjust weights of inputs.") ]
      let others-right? FALSE
      let me-right? FALSE

      ; flattening code taken from weigh-and-take-mode function, not "DRY"
      let val-from-others-input 0
      let others-input my-input-from-others
      if not empty? others-input [
        set others-input unnest others-input
        set val-from-others-input one-of modes others-input
      ]

      if (real-problem-val = val-from-others-input) [set others-right? TRUE ]

      let val-from-my-mem one-of modes my-mem

      if (real-problem-val = val-from-my-mem) [set me-right? TRUE ]

      let old-weight my-weight-others-input
      let increment-for-adj weight-adj-increment

      if (me-right? and not others-right?) [
        set my-weight-others-input (my-weight-others-input - increment-for-adj)
        if verbose? [ print (word "Turtle " who " here, adjusting others down." ) ]
      ]

      if (not me-right? and others-right?) [
        set my-weight-others-input (my-weight-others-input + increment-for-adj)
        if verbose? [ print (word "Turtle " who " here, adjusting others up." ) ]
      ]

      if (my-weight-others-input < 0) [ set my-weight-others-input 0 ]
      if (my-weight-others-input > 1) [ set my-weight-others-input 1 ]

    ]
    [
      if verbose? [ print (word "Turtle " who " here, I got it right, I'll leave my weights be." ) ]
    ]
  ]
end

; #################################
; ### The landscapes-map button ###
;
; When I created the landscape-map button I was a slightly better programmer than before
; writing the meat and potatoes of this ABM. Thus it uses functions more. I copied the
; original landscape generation code and then modified it to be more functional.
; Speed over elegance and refactoring, and I needed the model to work the same as v24,
; I wanted NO risk that somehow I would change the model behavior by implementing this.

; It's nuts, but the original code had each column as a lane and each row as a
; problem-set per tick. Thus I have to transpose in this modified code.

to landscapes-map
  ; loop here which calls generate-a-landscape

  ; define num-length and num-lanes here so I can pass them

  let lane-length 20
  let num-of-lanes 10
  let stagger-options [ 0 1 ]

  let oppositized-row-options [ 0 0.1 0.2 0.3 0.4 0.5 ]
  let ttl-length ((lane-length * (length oppositized-row-options)) - 1)
  let ttl-height ((num-of-lanes * (length stagger-options)) - 1  )
  resize-world 0 ttl-length 0 ttl-height
  if verbose? [ print word "resize-world (length height): " (list ttl-length ttl-height) ]

  let start-point [ 0 0 ]

  foreach stagger-options
  [
    n ->
    let i-counter 0
    foreach oppositized-row-options
    [
      ; make this one one huge row
      ; add to ttl-ld-matrix
      p -> if verbose? [ type word "p: " p print word " n: " n ]

      ; replace-item index list value
      set start-point replace-item 0 start-point (lane-length * i-counter)
      set start-point replace-item 1 start-point (num-of-lanes * n)
      if verbose? [ print word "start-point: " start-point ]

      ; generate-a-landscape [ cell-elements num-lanes num-length num-trans num-stagger ratio-opposite ]
      let my-ld generate-a-landscape ([1 2]) num-of-lanes lane-length 3 n p
      if verbose? [ print matrix:pretty-print-text my-ld ]
      if verbose? [ print word "foreach loop dims of matrix are: " matrix:dimensions my-ld ]
      viz-landscapes my-ld start-point
      set i-counter i-counter + 1
      if verbose? [ print word "i-counter: " i-counter ]
    ]

  ]

end

to viz-landscapes [ input-ld-matrix start-point ]
  let some-ld-matrix matrix:transpose input-ld-matrix
  let mat-dims matrix:dimensions some-ld-matrix
	let cols-count item 1 mat-dims
  let rows-count item 0 mat-dims
  let start-x item 0 start-point

  let start-y item 1 start-point
  if verbose? [ print word "start-y: " start-y ]
  if verbose? [ print word "start-y + rows-count: " (start-y + rows-count) ]

  if verbose? [ print word "mat-dims: " mat-dims ]

  let pselection patches with [
    (member? pxcor (range start-x (start-x + cols-count)))
     and
    (member? pycor (range start-y (start-y + rows-count)))
  ]

  if verbose? [ print word "patch selection: " pselection ]
  if verbose? [ ask pselection [print self] ]
  if verbose? [ print word "max patch: " max-n-of 1 pselection with [pxcor] [pycor] ]

  ask pselection [
    let my-val (matrix:get some-ld-matrix (pycor - start-y) (pxcor - start-x) )
    ifelse (my-val = 1) [set pcolor blue][set pcolor white]
  ]
end

to-report generate-a-landscape [ cell-elements num-lanes num-length num-trans num-stagger ratio-opposite ]
  ; Making the var names different to avoid problems and changing previous simulation code
  ; cell-elements - problem values which are the cells, should be list [ 1 2 ]
  ; num-lanes - basically the number of workers which is number of lanes
  ; num-length - basically the number of problems, length of the lanes
  ; num-trans - number of num-trans in the landscape between binary values
  ; stagger? - are the lanes offset by 1 from their neighbors or not
  ; ratio-opposite - ratio of total lanes where problems are inverted or "oppositized"

  ; I remove clean start because that's irrelevant to the image of landscapes
  ; let clean-start (10 + agent-mem-length)
  let clean-start 0

  let number-usable-rows round (num-length)

  let ld-matrix (matrix:make-constant num-length num-lanes
    (item 0 cell-elements))

  if verbose? [ print word "number-usable-rows: " number-usable-rows ]
  ;note: number-usable-rows round (num-length - clean-start)
  let nth-of-usable (floor (number-usable-rows / (num-trans + 1)))

  ; if it's not a clean division, get the remainder and append it to the last nth later
  let the-remainder remainder number-usable-rows (num-trans + 1)
  if verbose? [ print word "the remainder: " the-remainder ]

  if verbose? [ print(word "nth-of-usable: " nth-of-usable) ]
  let switch-row 0
  if verbose? [ print(word "switch-row: " switch-row) ]

  let trans-counter 0
  let colorit? TRUE
  let list-of-rows-to-change []
  while [trans-counter < num-trans]
  [
    if colorit? [
      let list-rows n-values (nth-of-usable) ; n-values size reporter

      ; Reports a list of length size containing values computed by repeatedly running the reporter.
      [ i -> i + (switch-row + (nth-of-usable * (trans-counter + 1))) ] ; move nth up to clean and lead start

      if verbose? [ print (word "list-rows: " list-rows) ]
      set list-of-rows-to-change (sentence list-of-rows-to-change list-rows)
      ; deal with the remainder, append it to the last nth

      if (the-remainder > 0 and (trans-counter = (num-trans - 1))) [
        if verbose? [ print "Remainder appending if-function is being called." ]
        let remainder-rows n-values (the-remainder)
        [i -> i + (switch-row + (nth-of-usable * (trans-counter + 2)))]
        if verbose? [ print word "Remainder rows: " remainder-rows ]

        set list-of-rows-to-change (sentence list-of-rows-to-change remainder-rows)
      ]
    ]
    ifelse (colorit?) [set colorit? FALSE] [set colorit? TRUE]
    set trans-counter trans-counter + 1
  ]

  if verbose? [ print (word "list-of-rows-to-change: " list-of-rows-to-change ) ]
  foreach list-of-rows-to-change [
    i -> matrix:set-row ld-matrix i (n-values num-lanes [(item 1 cell-elements)] )
  ]

  ifelse (num-stagger > 0 ) [
    set ld-matrix ld-desyncifier ld-matrix num-stagger clean-start cell-elements
  ]
  [
    if verbose? [ print "num-stagger was 0" ]
  ]


  ifelse ( ratio-opposite > 0 ) [
    if verbose? [ print word "ratio-opposite: " ratio-opposite ]
    if verbose? [ print "ratio-opposite > 0" ]
    set ld-matrix ld-oppositize ld-matrix num-lanes ratio-opposite clean-start
  ]
  [
    if verbose? [ print "ratio-opposite was 0 when evaled for conditional"]
  ]

  ; removing randomness from row chosen so I can group landscapes in the analysis phase
  ; I can't see how having random rows vs not oppositized will affect results

  if verbose? [ print ld-matrix ]
  if verbose? [ print matrix:pretty-print-text ld-matrix ]

  report ld-matrix

end

to-report ld-desyncifier [ some-matrix stagger-num clean-start cell-elements]
  let my-matrix some-matrix
  let the-dims matrix:dimensions some-matrix
  let cols-count item 1 the-dims

  ; USE AS NUMBER-OF-WORKERS BELOW

  ; Desync/stagger matrix.
  ; Make list that's index of every problem-matrix col, same as the number-of-workers.
  let col-index-list n-values cols-count [ i -> i ]
  if verbose? [ print (word "col-index-list: " col-index-list) ]

  let stagger-list map [ i -> i + stagger-num ] col-index-list
  if verbose? [ print (word "stagger-list: " stagger-list) ]

  foreach col-index-list [
    z -> let col-index z

    let prob-col-list matrix:get-column my-matrix col-index
    if verbose? [ print(word length prob-col-list " prob-col-list: " prob-col-list) ]

    ;make sublist without trash
    let trashless-list sublist prob-col-list clean-start (length prob-col-list)
    if verbose? [ print (word length trashless-list " trashless-list: " trashless-list ) ]

    ; make sublist
    let front-probs sublist trashless-list 0 (item col-index stagger-list)
    if verbose? [ print (word length front-probs " front-probs: " front-probs ) ]

    ;sentence the last bits back on
    let list-to-trim sentence trashless-list front-probs
    if verbose? [ print (word length list-to-trim " list-to-trim: " list-to-trim) ]

    ;trim the item x number of items off the front of the prob-col-list
    let new-col-list sublist list-to-trim ; provide first and second position below
      (item col-index stagger-list) (length list-to-trim)
    if verbose? [ print (word length new-col-list " new-col-list: " new-col-list) ]

    ; put the trash back on the front.
    let trash-list n-values clean-start [item 0 cell-elements]
    if verbose? [ print(word length trash-list " trash-list: " trash-list) ]

    let new-col-plus-trash sentence trash-list new-col-list

    ; Write back to matrix
    matrix:set-column my-matrix col-index new-col-plus-trash

  ]

  ; Verify that every column has the same number of values
  if verbose? [
        foreach col-index-list [y -> print(word "Sum of col " y " is "
          reduce + matrix:get-column my-matrix y )]
  ]

  report my-matrix

end

to-report ld-oppositize [ some-matrix num-lanes ratio-opposite clean-start]
  let my-matrix some-matrix
  let number-cols (round (ratio-opposite * num-lanes))
  if verbose? [ print(word "oppositizer number-cols: " number-cols ) ]

  ; make list of workers, choosing each item/worker at random an oppositizer number of times
  let agent-list n-of number-cols (n-values num-lanes [ i -> i])

  if verbose? [ print word "agent-list: " agent-list ]

  ; foreach item in that list
  foreach agent-list
    [ z ->
      let col-index z
      if verbose? [ print word "col-index: " col-index ]

      let prob-col-list matrix:get-column my-matrix col-index
      if verbose? [ print(word length prob-col-list " prob-col-list: " prob-col-list) ]

      let new-col []

      foreach prob-col-list
        [ x ->
          ifelse x = 1
          [
            if verbose? [ print(word "x: " x) ]
            set new-col lput 2 new-col
            if verbose? [ print(word length new-col " new-col: " new-col) ]
          ]
          [
            if verbose? [ print(word "x: " x) ]
            set new-col lput 1 new-col
            if verbose? [ print(word length new-col " new-col: " new-col) ]
          ]
        ]

      if verbose? [ print(word length new-col " new-col: " new-col) ]

    ; Write back to matrix
    matrix:set-column my-matrix col-index new-col

    ]

  report my-matrix
end


; #################################
; ### GO LOOP ###

to go
  reset-timer

  ; when the mean of the problem counters equals the length of the matrix in columns

  if (mean [my-problem-counter] of turtles) = length (matrix:get-column problem-matrix 0) [
    if verbose? [ print "Reached the end of the problem matrix." ]
    if verbose? [ print (word "Total score: " total-score) ]
    stop
  ]

  if verbose? [ print (word "-- COMMUNICATION PHASE -- ") ]

  communicate-among-agents

  if verbose? [ print (word "-- DECISION PHASE -- ") ]

  tackle-problem ; for workers AND managers
  ; includes these functions:
  ; for workers:
  ;   decide-solution
  ;   store-problem-in-memory ; stores the actual problem. could expand to use perception of prob
  ;
  ; for managers:
  ;   decide-solution ; uses the same strategy as the worker to consolidate the inputs into one
  ;   store-consolidation-in-memory

  score-solutions

  if (adj-input-weights = "uniform") [
    ask workers [
      let real-problem-val (matrix:get problem-matrix my-problem-counter who)
      adjust-weights-if-needed real-problem-val
    ]
    ask managers [
      let right-mode first my-mem
      adjust-weights-if-needed right-mode
    ]
  ]


  ask turtles [set my-problem-counter (my-problem-counter + 1)]
  ; set problems-solved-results sum [problems-solved] of turtles

  if visualization? [
    ask turtles [
      forward 1
    ]
  ]

  if verbose? [ type "Timer: " show timer ]
  if verbose? [ print (word "## TICK " ticks " ended.") ]
  tick
  if verbose? [ print (word "## TICK " ticks " ready to start...") ]

end
@#$#@#$#@
GRAPHICS-WINDOW
345
10
1883
79
-1
-1
6.0
1
10
1
1
1
0
1
1
1
0
254
0
9
1
1
1
ticks
30.0

SLIDER
9
14
181
47
number-of-workers
number-of-workers
1
10
10.0
1
1
NIL
HORIZONTAL

SLIDER
21
46
193
79
problems-per-worker
problems-per-worker
30
255
255.0
1
1
NIL
HORIZONTAL

CHOOSER
21
80
171
125
problem-type
problem-type
"binary"
0

CHOOSER
4
251
168
296
decision-strategy
decision-strategy
"mode with input"
0

BUTTON
10
212
73
245
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
648
450
809
605
11

BUTTON
132
212
187
245
go
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
75
212
130
245
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
11
333
182
366
weight-others-input
weight-others-input
0
1
0.75
.05
1
NIL
HORIZONTAL

SLIDER
11
298
183
331
agent-mem-length
agent-mem-length
0
9
9.0
1
1
NIL
HORIZONTAL

SLIDER
11
517
177
550
number-of-managers
number-of-managers
0
1
1.0
1
1
NIL
HORIZONTAL

CHOOSER
12
470
150
515
recall-type
recall-type
"most recent"
0

SWITCH
12
552
179
585
comm-connects-unlimited
comm-connects-unlimited
1
1
-1000

SLIDER
12
586
178
619
comm-connections-max
comm-connections-max
1
10
10.0
1
1
NIL
HORIZONTAL

TEXTBOX
210
603
541
648
mem-type \"freq table\" option not yet implemented\nproblem-type option not yet implemented\noppositizer is what % of cols to reverse symbols on
12
0.0
1

PLOT
214
448
414
598
Total score over ticks
ticks
score
0.0
100.0
0.0
100.0
false
false
"" ""
PENS
"pen-1" 1.0 0 -5298144 true "" ";plot total-score"

CHOOSER
3
125
195
170
env-pattern
env-pattern
"custom"
0

PLOT
215
294
843
444
Per-tick score over ticks
NIL
score per tick
0.0
100.0
0.0
100.0
false
false
"" ""
PENS
"pen-1" 1.0 0 -13345367 true "" ";plot tick-score"

SLIDER
22
171
194
204
oppositizer
oppositizer
0
.50
0.5
.05
1
NIL
HORIZONTAL

SWITCH
648
207
754
240
verbose?
verbose?
1
1
-1000

SWITCH
648
241
776
274
visualization?
visualization?
0
1
-1000

TEXTBOX
15
454
198
482
What info from mem to transfer:
11
0.0
1

CHOOSER
11
366
149
411
adj-input-weights
adj-input-weights
"off" "uniform"
1

SLIDER
11
411
183
444
weight-adj-increment
weight-adj-increment
0
1
0.5
.05
1
NIL
HORIZONTAL

MONITOR
648
160
721
205
NIL
total-score
17
1
11

INPUTBOX
199
125
261
185
transitions
9.0
1
0
Number

INPUTBOX
263
125
317
185
stagger
1.0
1
0
Number

TEXTBOX
200
185
350
214
max transitions = 9\n0 or 1 for stagger
11
0.0
1

BUTTON
346
155
466
188
NIL
landscapes-map
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

The basic research question was whether hierarchies were useful in changing environments, and if so, when, how, and why?

In this model, a simple hierarchical organization attempts to adapt to a changing environment, and where that organization consists of workers and managers that see information at different scales: ground-level vs higher-level. 

The primary design focus was on organizational learning emerging from individual learning and communication. The secondary design focus was exploring whether different environments performed differently with the same organizational structures, leading to tradeoffs.

## HOW IT WORKS

The model has local environments, agents, and basic communication.

Environment:
We create a grid-cell environment that is simple while still making it possible for agents to experience different local environments in space and time. We first create a strip of cells where each one worker stands on one cell. Each worker’s cell is their local state: that state is one of two binary values, and can change its value on each time step of the model. Static environments are of no interest, so the simplest environment in our set is a synchronous environment where all local states are the same in space and change their values simultaneously over time. These environments have little complexity, since every lane is a duplicate of its neighbor lanes. To create more complex environments, we modify the landscape by delaying or inverting the onset of environmental changes the agents experience relative to one another.

Each cell “problem” is a 0 or 1; each worker has its own lane of problems. When landscapes are synchronous, problem transitions happen across lanes simultaneously. When landscapes are asynchronous, that asynchrony is controlled by two parameters: 1) whether problems are staggered (delayed in transition) among the lanes, so that the transition hits each worker at a different time; 2) what percentage of lanes are made the inverse of other lanes. Staggering transitions among workers is a milder form of asynchrony, whereas inverted rows represent very different local sub-environments from the overall environment. By varying these parameters we can create increasingly asynchronous environments with either delayed environmental changes across agents, or some local conditions that are the inverse of the overall environment, or both.

Agents:
In this model, each agent must solve one problem per round by correctly guessing the incoming local environmental state. Agents–both workers and managers–have no ability to foresee future problems, so they must rely on their memory of past problems and on input from others. Agents have memories which store the last n problems they have seen (their memories are 3 slots to 9 slots long). When deciding on a strategy for the next problem, they consult their memory and any inputs (i.e., advice from the manager) and take the mode of that set of values. Agent capabilities and preferences are homogeneous within runs, but become varied in their memories and choices as they learn their heterogeneous environment. The agents’ decision-making is focused only on getting the next tick right.
Agents have input weights that are a 0 to 1 ratio of how much weight their put on their memory vs their manager’s input (assuming they’re in the manager condition). Since the choice is the mode, the inputs are multiplied to give the mode a set of inputs that are the correct ratio for that weight. Agents also have weight adjustment increment settings, which at 0 mean that they cannot adjust the input weight value, but above 0 means they can adjust the weight by that value each tick (e.g. 0.2 means that if the agent decides to adjust the ratio, they will change it by 0.2 only, not more or less).


### Setup/Initialization

Set-up environment 

1.	Set-up the problem matrix/grid according to the environmental input parameters.

Set-up agents

1.	Create a worker class, and, if the run has a manager, a manager class.

2.	Create the number of workers and managers specified in the input parameters.

### Run loop

1.	If and only if the run has a manager:

	a.	Each worker communicates their last problem to the manager.

	b.	The manager communicates the mode of the last problem to the workers as input.

2.	Each worker uses its memory and manager input (if enabled for the run), weighed according to its current variable value, to calculate the mode. This is its guess for the current problem.

3.	The manager (if enabled for the run) uses its memory and worker input, weighed according to their current variable value, to calculate the mode. This is its guess of the current problem (which it will communicate to the workers the next time step). 

4.	Each worker and the manager store their guess in their memory.

5.	The true value of the current problem is revealed to each worker, who stores it in memory. The true mode of the current problem is revealed to the manager (if the run has a manager).

6.	Each worker and the manager (if enabled) scores whether their guess for this time step was correct.

7.	Each worker updates their score with whether they were correct. The manager does not contribute to the score.

8.	If future problems remain on the grid, agents move forward to the next time step and set of problems; loop to Run step 1.

### Group scoring

1.	When the run has finished (there are no future problems remaining), do not count scores of early time steps, since the agents’ memories will not be filled. The first n problems, where n = 10 + the agents’ memory length, should not be counted in the score.

2.	Sum the scores for all workers in the group to calculate the group score.


## HOW TO USE IT

This model follows the standard sequence: click "setup", click "go". However, the model is meant primarily to compare how well different groups perform, so BehaviorSpace should be used to generate data that enables comparing parameter sets.

### Interface settings

number-of-workers: The number of workers in the simulation. Does not include the manager in the count.

problems-per-worker: The number of problems, which are cells, one encountered per time step.

problem-type: Binary uses 0s and 1s. The model can be extended to use a larger set of categories.

env-pattern: The model has preset patterns. The custom setting provides the most control over the landscape. Using custom enables the number of transitions (0 to 1, or 1 to 0), whether the landscape is staggered in a stairstep pattern, and whether any lanes are inverted, i.e. whether in a lane the 0s become 1s and the 1s become 0s. 

transitions: The number of times each lane switches. E.g. 0 to 1 is one transition, and then 1 to 0 is another transition.)

stagger: Whether the columns of problems are delayed in a stairstep pattern.

oppositizer: The ratio of lanes inverted. A setting of .5 would result in half the lanes switching their values: the 0s become 1s and the 1s become 0s. 

decision-strategy: The default setting is the mode with input strategy.

agent-mem-length: The number of items each agent's memory can store. The memory stores previous problems, starting with the most recent.

weight-others-input: The weight each agent puts on input from the manager. The rest of the weight is put on the agent's memory. For example, if this value is at .25 then the agent will put 25% weight on the manager's input and 75% weight on their own memory. If the run has the weight-adj-increment setting at greater than 0, then the agent can adjust this weight variable based on the performance of the manager's input. If there's no manager enabled for the run, this setting does not matter for the run.

adj-input-weights: This setting controls the distribution for the input weights. The model can be extended to use other distributions, but the default is uniform.

weight-adj-increment: This value is how much the input weight is adjusted in a step, if it is adjusted. E.g., if weight-others-input is .75 and this variable is .25, and if the manager gets the time step wrong and the agents memory gets it right, the weight-others-input will be reduced .25 to .5.

recall-type: This controls which memory item the agents communicate. Recent is the default, and uses the last, most recent, memory item.

number-of-managers: Runs can have managers or not, this can be set to only 0 or 1.

comm-connects-unlimited: This enables workers and managers to have unlimited communications to one another, versus having a restriction on how many workers the manager can talk to each time step.

comm-connections-max: This is the maximum number of connections the workers and managers can have. The default is 10 connection, which means that the manager can talk to all 10 or less workers every timestep. Less than 10, with 10 workers, means that some workers will not receive information every time step.

landscapes-map button: This creates a visualization of landscapes to show how transitions, stagger, and oppositizer settings would affect a basic landscape. The resulting landscape is for visualization only, the agents cannot use it in a run.

verbose?: This outputs text to the command center for troubleshooting. Using it slows down runs considerably, so it should usually remain off.

visualization?: This places agents on the landscape and moves them with each tick. This slows down runs considerably, but is useful to see what the agents are encountering.


## THINGS TO NOTICE

Hierarchy has advantages on simple, synchronous landscapes. As landscapes get more complex, hierarchy may not be as good as each agent paying attention to only their local condition. However, giving agents the ability to adjust the relevance of information, using the weight-adj-increment, improves performance in hierarchical groups.


## THINGS TO TRY

Generally the way to use the model is to compare how well a group with a manager does compared to a group without one.

1. Try a simple landscape with transitions at 9 but with stagger at 0 and oppositizer at 0. Compare the scores of a group without a manager and a group with a manager.

2. Try more challenging landscapes and see how much difference a manager makes.

3. Try setting weight-adj-increment to 0 so that the agents can change their weights. Do they do better or worse than groups that can adjust their weights?

## EXTENDING THE MODEL

The model has a modular design so that it can be extended to simulate more sophisticated behavior. Possible enhancements include:

1. New decision strategies, such as a strategy that weighs more recent information more heavily in decision-making.

2. Enabling workers to communicate with each other (the adjacency matrix in the model provides support for this).

3. Making heterogeneous agent populations possible, e.g., ones where agents have varied memory lengths.

4. Including noise in the model where communication has a chance of error.

5. Enabling the model to use more categories of problem than 0 and 1. Note: this would require substantial new code.

## CREDITS AND REFERENCES

The model was analyzed for the paper "An agent-based model of hierarchical information-sharing organizations in asynchronous environments"
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="adjusted weights test 01" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;off&quot;"/>
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="with and without oppositizer" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <enumeratedValueSet variable="number-of-workers">
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;switch-then-back-sync&quot;"/>
      <value value="&quot;switch-then-back-no-sync&quot;"/>
      <value value="&quot;switch-back-twice-sync&quot;"/>
      <value value="&quot;switch-back-twice-no-sync&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="76"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="modify size to be able to run" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <enumeratedValueSet variable="number-of-workers">
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;switch-then-back-sync&quot;"/>
      <value value="&quot;switch-then-back-no-sync&quot;"/>
      <value value="&quot;switch-back-twice-sync&quot;"/>
      <value value="&quot;switch-back-twice-no-sync&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="76"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="modify size to be able to run x10" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <enumeratedValueSet variable="number-of-workers">
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;switch-then-back-sync&quot;"/>
      <value value="&quot;switch-then-back-no-sync&quot;"/>
      <value value="&quot;switch-back-twice-sync&quot;"/>
      <value value="&quot;switch-back-twice-no-sync&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="76"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="test landscape reporter" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <metric>landscape</metric>
    <enumeratedValueSet variable="number-of-workers">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;switch-then-back-sync&quot;"/>
      <value value="&quot;switch-then-back-no-sync&quot;"/>
      <value value="&quot;switch-back-twice-sync&quot;"/>
      <value value="&quot;switch-back-twice-no-sync&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="76"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="adjusted weights test 02" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;off&quot;"/>
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="2"/>
      <value value="6"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="adjusted weights w landscapes 01" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <metric>landscape</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;off&quot;"/>
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="3"/>
      <value value="5"/>
      <value value="7"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="adjusted weights w landscapes 02" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <metric>landscape</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;off&quot;"/>
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="adjusted weights w landscapes 03" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <metric>landscape</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="adjusted weights w landscapes 04 x 10" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <metric>landscape</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="adjusted weights w landscapes 04 x 20" repetitions="20" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>total-score</metric>
    <metric>landscape</metric>
    <enumeratedValueSet variable="problems-per-worker">
      <value value="255"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-workers">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="problem-type">
      <value value="&quot;binary&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-pattern">
      <value value="&quot;custom&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="transitions">
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stagger">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="oppositizer">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="decision-strategy">
      <value value="&quot;mode with input&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recall-type">
      <value value="&quot;most recent&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-others-input">
      <value value="0"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adj-input-weights">
      <value value="&quot;uniform&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-adj-increment">
      <value value="0"/>
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
      <value value="0.4"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="agent-mem-length">
      <value value="3"/>
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
      <value value="9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connections-max">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comm-connects-unlimited">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-managers">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
