extensions [ r array matrix nw ]

globals [ ;; action-performance-matrix

          Q-action-performance-matrix   ;; a matrix of N * 2^(1 + K + C*S) uniformly distributed random numbers,
                                        ;; which represents the performance contribution of given actions.

          nk-action-connections-matrix
          nc-action-connections-matrix

          when-performance-unchange     ;; the simulation time when convergency.

          avg-performance               ;; average performance of the community of agents.

          avg-actions-syn               ;; average value of the cosine similarity between actions-list of agents.
             ]

turtles-own [ action-interdependency-matrix ;; matrix representing the interdependency of actions.

              nk-other-actions-matrix   ;; interdependency of actions within the agent

              ns-other-agents-set
              nc-other-actions-matrix   ;; interdependency of actions across agents.

              current-actions-array     ;; action sequences the agent carries out at present.
              next-actions-array        ;; action sequences the agent will adopt at the next simulation tick.


              my-action-performance-array

              my-performance            ;; performance value (e.g., fitness) of the agent.

              explore                   ;; a boolean variable to represent the agent's exploration status,
              exploit                   ;; a boolean variable to represent the agent's exploitation status,
                                        ;; 0 - failed; 1 - succeed.

              total-num-explore         ;; total number of explorative activities occurring
              total-num-exploit         ;; total number of exploitative activities occurring
                       ]


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; the setup procedure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  clear-all
  ask patches [ set pcolor white ]

  create-Q-action-performance-matrix
  ;; call the 'create-action-performance-matrix' procedure.
  ;; set action-performance-matrix create-action-performance-matrix N K C S


  ;  print "============================================================================="
  ;  print "This is the action-performance-matrix: "
  ;  print matrix:pretty-print-text Q-action-performance-matrix
  ;  print "-----------------------------------------------------------------------------\n"

  create-action-connections-matrix

  ;; call the 'create-agents' procedure to create the community of agents with given network structure.
  create-agents

  ;; initializing parametic settings
  initialize-agent-status

  ;; set the average performance value in a global variable.
  set avg-performance mean [my-performance ] of turtles

  reset-ticks
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; the main procedure
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go

  ;; conditions to end the program.
  if ( ( ticks >= end-time ) or ( ticks - when-performance-unchange > 500 ) ) [ stop ]
  ;  if (  ticks >= end-time  ) [ stop ]

  ;; update agents' status
  ask turtles [
    ; if ( exploit? or explore? ) [
    ; show word "current-actions " current-actions-array
    ; show word "next-actions " next-actions-array

    set current-actions-array array:from-list (array:to-list next-actions-array )
    set my-performance caculate-performance current-actions-array

    set total-num-explore total-num-explore + explore
    set total-num-exploit total-num-exploit + exploit

    set explore 0
    set exploit 0
  ]


  ;; caculate the synchroney level of actions, and update the variable value
  set avg-actions-syn caculate-avg-actions-syn

  ;; check if the routines system reaches a stable status, update parameter values.
  if abs ( avg-performance - mean [my-performance ] of turtles ) > 0.0001 [
    set when-performance-unchange ticks
    set avg-performance mean [my-performance ] of turtles ]

  ; show word "action synchrony between turtles " avg-actions-syn

  ;  ]

  ;; randomly choose an agent, exacute the exploitation or exploration activities.
  ask one-of turtles [

    if (my-performance - min [my-performance ] of link-neighbors < 0.0001) [
    if (is-exploit?  ) [
      exploit-activity ]

    if (is-explore? and ( exploit = 0 ) ) [
      explore-activity  ]

    ]
  ]


  tick
end


;; this is the main procedure for one time step
to go-once
  go

end


;; to exploit procedure
to exploit-activity

  ; set exploit? false
  ; print "to exploit: ---------------"
  ; show word "current-actions-list " current-actions-array


  ;; the exploitative activity,
  ;; randomly choose an action, change its value from 1 to 0 or vice versa.
  let a-list array:to-list current-actions-array

  let a-posi random N
  set a-list replace-item a-posi a-list ( 1 - array:item current-actions-array a-posi )

  ; show word "value at the posi " ( 1 - array:item current-actions-array a-posi )
  ; show word "a-position " a-posi
  ; show word "a-list " a-list
  ; show word "current-actions " current-actions-array

  let a-array array:from-list a-list

  let a-performance caculate-performance a-array
  if ( a-performance - my-performance > 0.0001) [
    set next-actions-array a-array
    set exploit 1 ]

  ; show word "nextactions-list " next-actions-array
  ; show word "exploit? " exploit
  ; print "-------------------------------------------\n"

end

;; to explore
to explore-activity

  ; set explore? false

  ;; the explorative activity,
  ;; randomly select a neighbor, copy chunk of bitstrings.

  let a-neighbor one-of link-neighbors
  let a-list array:to-list current-actions-array
  let b-list array:to-list [current-actions-array ] of a-neighbor

  ; show word "a-list " a-list
  ; show word "b-list " b-list

  if (a-list != b-list ) [
    let ii 0
    repeat N [
      if ( ( item ii a-list != item ii b-list ) and random-float 1 < ( 1 - p-error ) ) [
        set a-list replace-item ii a-list item ii b-list ]
      set ii ii + 1
    ]

    let a-array array:from-list a-list
    let a-performance caculate-performance a-array
    if ( a-performance - my-performance > 0.0001 ) [
    ; show word "x-list " a-list
    set next-actions-array array:from-list (array:to-list a-array )
    set explore 1 ]

  ]


  ; show word "expore? " explore
  ; print "-------------------------------------------\n"

end


to create-Q-action-performance-matrix
  ;; this procedure uses the method described in rmnkGenerator.R through r extensions
  ;; to create the ρMNK-Landscapes
  ;; see details in: http://mocobench.sourceforge.net/index.php?n=Problem.RMNK

  r:put "rho" rho
  r:put "Q" Q
  r:put "N" N
  r:put "K" K
  r:put "C" C
  r:put "S" S

  r:eval "R <- matrix(rep(rho, Q), Q, Q) "

  r:eval "diag(R) <- rep(1, Q)"
  r:eval "RR <- 2*sin(pi/6 * R)"
  r:eval "data <- data.frame(pnorm(MASS::mvrnorm(n = N * 2 ^ (C * S + K + 1), mu = rep(0, Q), Sigma = RR )) )"

  set Q-action-performance-matrix matrix:from-row-list r:get "data"
  ;; show word "data from r (Q-action-performance) " Q-action-performance-matrix

end


;;; to create the matrix of N * 2^(1 + K + C*S) uniform distributed random numbers.
;to-report create-action-performance-matrix [ nn kk cc ss ]
;  random-seed 100
;  let dd 2 ^ ( 1 + kk + cc * ss )
;   report matrix:from-row-list n-values nn [ n-values dd [ random-float 1 ] ]
;
;end


;; this procedure creates the matrix representing the NKCS connections of invidual actions
to create-action-connections-matrix
  ;set nk-action-connections matrix []
  ;set nc-action-connections matirx []

  ;; when K > 0, create the nk connections.
  if ( K > 0 ) [
    set nk-action-connections-matrix matrix:make-constant N K 0
    let ii 0
    repeat N [
      ;; a temp list
      let k-actions-location-list []
      ;; create k other actions' locations, and store those values in k-actions-location-list
      repeat K [
        let jj 0
        let action-location random N
        ;; the k actions should be different with eachother
        while [ ( member? action-location k-actions-location-list or action-location = ii ) and jj < 50] [
          set action-location random N
        set jj jj + 1 ]

        set k-actions-location-list lput action-location k-actions-location-list ]

        ;; put the k actions' location info. into the nth row of matrix.
        matrix:set-row nk-action-connections-matrix ii k-actions-location-list

      set ii ii + 1 ] ]

  ; print the results.
  ; show word "nk-action-connections " nk-action-connections-matrix


  if (C > 0 ) [
  set nc-action-connections-matrix matrix:make-constant N C 0
  let ii 0
  repeat N [
    ;; a temp list
    let c-actions-location-list []

    repeat C [
      let jj 0
      ;; create c other actions' locations, and store those values in c-actions-location-list
      let action-location random N
      set c-actions-location-list lput action-location c-actions-location-list ]

    ;; put the s actions' location info. into the nth row of matrix.
    matrix:set-row nc-action-connections-matrix ii c-actions-location-list

    set ii ii + 1 ] ]

    ;  show word "nc-action-connections " nc-action-connections-matrix
    ;    print "\n"
end


;; this procedure creates the community of agents with given network structure.
to create-agents

  ;; create Q agents.
  create-turtles Q [
    set color orange
    set shape "circle" ]

  ;; create links with different network structures.
  if (network-structure = "star-shape") [
    ask one-of turtles [
      ;; set color blue
      create-links-with other turtles ]
  ]

  if (network-structure = "full-connected") [
    ;; nw:generate-star turtles links 10
    ask turtles [create-links-with other turtles ]
  ]

    if (network-structure = "random") [
    ask links [die ]
      ask turtles [
        create-link-with one-of other turtles ]
  ]

  ;; layout the agents on the world
  layout-circle turtles 15

end


to initialize-agent-status

  ;; initialize current-actions-list with random values 0 and 1.

  ; print "current-actions-list of agents: ----------------------------------------------"

  ask turtles [
    set current-actions-array array:from-list n-values N [ random 2 ]

    ;; initialize next-actions-list equals to current-actions-list values.
    set next-actions-array array:from-list (array:to-list current-actions-array )

    set exploit 0
    set explore 0

    ; print out the current-actions-list to concel
    ; show current-actions-array

  ]

  ; print "-----------------------------------------------------------------------------\n"

  ask turtles [
    set nk-other-actions-matrix nk-action-connections-matrix

    ;; when S > 0, create the ns-connections.
    if ( S > 0 ) [
      ;; randomly select the other C agents.
      set ns-other-agents-set n-of S other turtles
    ]

    ;; when C > 0, create the ns-connections.
    set nc-other-actions-matrix nc-action-connections-matrix
  ]

  ;; print to the consel
  ask turtles [
    ;   show word "nk-other actions: " nk-other-actions-matrix

    ;    if ( any? nc-other-agents-set ) [
    ;      print  "nc-other-agents: "
    ;
    ;      foreach sort nc-other-agents-set [? ->
    ;        show ? ]
    ;    ]

    ;    show word "nc-other-actions: " nc-other-actions-matrix
    ;    print "--------------------------------------------"

    ;    set my-performance caculate-performance current-actions-array

    set total-num-explore 0
    set total-num-exploit 0
  ]


  ( foreach ( sort turtles ) ( matrix:to-row-list Q-action-performance-matrix ) [ [ agent p-list ] ->
    ask agent [
      set my-action-performance-array array:from-list p-list
      ;    show word "my-action-performance " my-action-performance-array
    ]

  ] )

  ; print "--------------------------------------------\n"


end

;; this procedure caculates the value of synchroney between two agents.
to-report caculate-actions-syn [ agent1 agent2 ]
  let a1 0
  let b1 0
  let ab1 0
  let syn 0

  (foreach array:to-list [current-actions-array ] of agent1 array:to-list [ current-actions-array ] of agent2 [ [ ?1 ?2 ] ->
    set ab1 ab1 + ?1 * ?2
    set a1 a1 + ?1 ^ 2
    set b1 b1 + ?2 ^ 2

  ])

  if ( a1 * b1 > 0 ) [
    set syn ab1 / sqrt ( a1 * b1 )]
  report syn

end


;; this procedure caculates the average value of the variable avg-actions-syn.
to-report caculate-avg-actions-syn
  let agent-list sort turtles
  let sum-actions-syn 0

  let ii 0
  repeat Q - 1 [
    let jj ii + 1
    repeat Q - ii - 1 [
      set sum-actions-syn sum-actions-syn + caculate-actions-syn ( item ii agent-list ) (item jj agent-list )
      set jj jj + 1
    ]

  set ii ii + 1
  ]

  report sum-actions-syn * 2 / ( Q * ( Q - 1 ))
end

;; this procedure caculates the performance of a list of actions
;; according to the NKCS landscape.
to-report caculate-performance [ a-array ]
  let action-index 0
  let performance0 0

  ; show word "current-actions-list" current-actions-list

  let ii 0
  repeat N [
    ;; for each action in the current-actions-list
    set action-index array:item a-array ii

    ; show word "current-action-location " ii
    ; show word "current-action " action-index

    if ( K > 0 ) [
      let jj 0
      repeat K [
        let k-action-location matrix:get nk-other-actions-matrix ii jj
        set action-index action-index * 2 + array:item current-actions-array k-action-location

        ; show word "k-location " k-action-location
        set jj jj + 1 ] ]
        ;; show word "action-index " action-index

    if ( C * S > 0 ) [
      foreach sort ns-other-agents-set [ ?1 ->
        let actions-array [current-actions-array ] of ?1

        let gg 0
        repeat C [
          let c-action-location matrix:get nc-other-actions-matrix ii gg
          set action-index action-index * 2 + array:item current-actions-array c-action-location

          ; show word "ns-agents " [who ] of ?1
          ; show word "c-location " gg
          set gg gg + 1 ] ]
    ]

    ; show word "action-index " action-index
    ; show word "performance contribution "  matrix:get action-performance-matrix ii action-index
    ; print ""
    set performance0 performance0 + array:item my-action-performance-array ( ii * 2 ^ ( 1 + K + C * S ) + action-index )
                                    ;; matrix:get action-performance-matrix ii action-index

    set ii ii + 1 ]


    ;; define my-performance equals to the average of all actions' performance contribution.
    set performance0 performance0 / N

    ; show word "my-performance value: " performance0
    ; print "=============================================================================\n\n"

  report performance0

end



;; copyright @ Dehua Gao (dhuagao@gmail.com)

@#$#@#$#@
GRAPHICS-WINDOW
414
34
859
480
-1
-1
13.242424242424242
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
29
440
111
473
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
140
440
221
473
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
31
76
203
109
N
N
1
20
8.0
1
1
NIL
HORIZONTAL

SLIDER
32
117
204
150
K
K
0
N - 1
3.0
1
1
NIL
HORIZONTAL

SLIDER
32
162
204
195
C
C
0
N
1.0
1
1
NIL
HORIZONTAL

SLIDER
32
205
204
238
S
S
0
Q - 1
1.0
1
1
NIL
HORIZONTAL

SLIDER
31
35
203
68
Q
Q
1
20
10.0
1
1
NIL
HORIZONTAL

CHOOSER
33
358
206
403
network-structure
network-structure
"random" "star-shape" "full-connected"
2

BUTTON
250
440
330
473
NIL
go-once
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
942
40
1417
194
plot-performance
simulation tick
avg-performance
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"avg-performance" 1.0 0 -2674135 true "" "plot mean [my-performance ] of turtles"
"min-performance" 1.0 0 -7500403 true "" "plot min [my-performance ] of turtles"
"max-performance" 1.0 0 -1184463 true "" "plot max [my-performance ] of turtles"

MONITOR
1285
363
1419
408
avg-performance
avg-performance
6
1
11

PLOT
943
363
1268
533
plot explore-exploit
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"explore-num" 1.0 0 -4079321 true "" "plot mean [ total-num-explore ] of turtles"
"exploit-num" 1.0 0 -14454117 true "" "plot mean [ total-num-exploit ] of turtles"

INPUTBOX
240
338
333
398
end-time
10000.0
1
0
Number

MONITOR
1284
422
1420
467
simu-tick-to-steady
when-performance-unchange
6
1
11

SWITCH
238
197
374
230
env-change?
env-change?
1
1
-1000

SLIDER
33
249
205
282
rho
rho
0
1
0.9
0.01
1
NIL
HORIZONTAL

SLIDER
34
298
206
331
p-error
p-error
0
1
0.01
0.01
1
NIL
HORIZONTAL

SWITCH
239
64
373
97
is-explore?
is-explore?
1
1
-1000

SWITCH
239
108
373
141
is-exploit?
is-exploit?
0
1
-1000

MONITOR
1284
490
1420
535
avg-actions-syn
avg-actions-syn
6
1
11

PLOT
943
203
1417
353
actions-syn
NIL
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"syn" 1.0 0 -16777216 true "" "plot avg-actions-syn"

@#$#@#$#@
## WHAT IS IT?

This is an agent-based model to simulate the intricate relationship between artifacts-based explorative and exploitative activities carried out by multiple human actors on the one hand, and the interactive process between artifacts and the performance of routines – i.e., the ‘framing, overflowing and reframing’ cycle (D'Adderio, 2008; 2011) – on the other. 

By formalizing the internal structure and processes underpinning the formation dynamics of organizational routines, the model focuses on the artifacts-based exploration and exploitation at the micro level and investigates the collective outcome of patterned routines from the ‘bottom-up’. 



## HOW IT WORKS

In our setting, the organizational activity is performed simultaneously by Q agents within a community. Each agent q (where 1 ≤ q ≤ Q) faces an organizational task that can be expressed as the vector of N component actions.We employ the NKCS model (Kauffman and Johnsen, 1991; Levitan, et al., 2002) to formalize the interdependencies between individual actions as that: for each agent q, the performance contribution of action a(q,n) (where 1 ≤ q ≤ Q; 1 ≤ n ≤ N) not only depend on this action’s own status, but also is affected by, (1) K other actions of the agent q itself, and (2) C actions assigned to S other agents.

The algorithm of ρMNK-landscapes (Verel, et al., 2013) is used to generate Q landscape matrixes – i.e., each landscape matrix for one agent within the community, modeling the similarity of agents and the closeness between task environments of those similar agents within the community. 

The agent with competitive pressure (e.g., owns the smallest performance value among its neighbors) try to achieve exploitation based on its understanding and interpretation against existing artifacts. Exploitation here implies that the agents just locally search for alternative solutions and move towards the optimal solution step-by-step due to their myopia. For example, an agent randomly selects only one bitstring of its actions list at a time, and vary the bitstring’s value from 0 to 1 or vice versa. If the agent finds the alternative solution superior to its current status (i.e., with a better performance value h), it then adopts such an alternative solution at the next simulation tick.

Otherwise, the exploitation fails, and the agents explore through a distant or long jump by organizational learning – i.e., they communicate with and imitate from partners either consciously with some strong incentives or not. 

In this paper, we consider the impact of power asymmetry among interaction partners on routine dynamics, and formalize interpersonal interrelationships within the community in accordance with different types of network structures. We primarily use random network as the benchmark, and then consider two different networks including the Star-shape and Fully-connected network models – representing vertical (supervisor-subordinate) and horizontal (coworker) organizational structures, respectively. These two types of organizational structures correspond with the two ends of the spectrum of formal power distribution from high to low levels of power asymmetry.



## HOW TO USE IT

Items in the interface tab are that:
Q: Number of agents in the community, each with one organizational task to cope with iteratively;
N: Number of component actions required to accomplish an organizational task;
K: Internal coupling degree of the organizational task;
[C,S]: External coupling degree between organizational tasks;
rho: Similarity of the task environments between agents;
p-error: Probability of errors occurring in imitation between agents;
network-structure: Network structure that represents the interpersonal interrelationship of agents within the community;
[is-explore?, is-exploit?]: Boolean variables that determine whether exploration and exploitation activities occur (TRUE) or not (FALSE);
env-change?: Boolean variable to represent whether the envoronmental change (that is, the landscape) is allowed or not during the simulation. [to be extended]

Simulation outputs include:
(1) The average performance;
(2) The average synchroney level of individual actions;
And (3) the total numbers of explorative and exploitative activities occurring.

## THINGS TO NOTICE
Occasionally, we find that the simulation may not reach a stable status -- that is, with some or all simulation outputs still fluctuates by the end time. Such runs should be omitted when analyzing the simulation results.

## THINGS TO TRY

Users can also try to vary rho values from 0.5 to 1.0 to investigate its effect on the formation dynamics of organizational routines over time.

## EXTENDING THE MODEL

The impact of environmental change (e.g., env-change = TRUE) on routines formation dynamics can be considered in an extention of the model.


## NETLOGO FEATURES

Extensions including r, array, matrix, and nw are used in our Netlogo codes.

Note: to use the r extension, users need a compatible R installation and also to configure the extension. A reference is: Thiele, JC; Grimm, V (2010). NetLogo meets R: Linking agent-based models with a toolbox for their analysis. Environmental Modelling and Software, 25(8), 972-974

The version information of softwares we use here is: NetLogo 6.2.2 and R 4.2.0.

## RELATED MODELS

none.

## CREDITS AND REFERENCES

Citations: Gao, D., & Yang, Y. (2023). Identifying the impact of artifacts-based exploration and exploitation on routine’s formation dynamics: An agent-based model. Working paper.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>avg-performance</metric>
    <metric>when-performance-unchange</metric>
    <enumeratedValueSet variable="Q">
      <value value="4"/>
      <value value="8"/>
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K">
      <value value="0"/>
      <value value="1"/>
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="S">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structure">
      <value value="&quot;star-shape&quot;"/>
      <value value="&quot;full-connected&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-time">
      <value value="50000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change-period">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_benchmark_test0.1" repetitions="50" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>avg-performance</metric>
    <metric>when-performance-unchange</metric>
    <metric>avg-actions-syn</metric>
    <metric>sum [total-num-explore ] of turtles</metric>
    <metric>sum [ total-num-exploit ] of turtles</metric>
    <enumeratedValueSet variable="Q">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
      <value value="10"/>
      <value value="12"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K">
      <value value="0"/>
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="S">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rho">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p-error">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structure">
      <value value="&quot;random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-time">
      <value value="8000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change-period">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_benchmark_test0.2-3" repetitions="150" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>avg-performance</metric>
    <metric>avg-actions-syn</metric>
    <enumeratedValueSet variable="Q">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="S">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rho">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p-error">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structure">
      <value value="&quot;random&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-time">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="is-exploit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="is-explore?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change-period">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_scenario1" repetitions="85" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>avg-performance</metric>
    <metric>avg-actions-syn</metric>
    <enumeratedValueSet variable="Q">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K">
      <value value="0"/>
      <value value="1"/>
      <value value="3"/>
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="S">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rho">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p-error">
      <value value="0"/>
      <value value="0.01"/>
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structure">
      <value value="&quot;full-connected&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-time">
      <value value="6000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="is-exploit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="is-explore?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change-period">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment_scenario1_summary" repetitions="100" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>avg-performance</metric>
    <metric>avg-actions-syn</metric>
    <enumeratedValueSet variable="Q">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="N">
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K">
      <value value="0"/>
      <value value="1"/>
      <value value="3"/>
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="S">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="rho">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="p-error">
      <value value="0"/>
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-structure">
      <value value="&quot;star-shape&quot;"/>
      <value value="&quot;full-connected&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="end-time">
      <value value="10000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="is-exploit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="is-explore?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change-period">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="env-change?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
